/*
 * ssi_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
 
/******************************************************************************
 *
 * The vSSITaskCreate task configures the SSI peripherals of the TM4C123GH6PM
 * MCU.  SSI0 is configured as a Master and SSI2 is configured as a slave.  The
 * SSI clock is set to be 2MHz, the transfer size is 8-bits, and SSI Mode 0 is
 * used for the polarity and phase.  The pins used are documented in the
 * individual configuration functions.
 *
 * prvSSITransmitTask loads both SSI peripherals with data to send.  To
 * demonstrate the full-duplex capability of the SSI bus, the data will be sent
 * simultaneously.  Therefore the SSI2 peripheral is loaded with data first as
 * the SSI0 peripheral will drive the SSI clock.
 *
 * prvSSIReceiveTask handles receiving the SSI data that was transmitted from
 * both FIFO's and printing the data out via UART to display on a terminal
 * window.
 *
 * This example uses UARTprintf for output of UART messages.  UARTprintf is not
 * a thread-safe API and is only being used for simplicity of the demonstration
 * and in a controlled manner.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/ssi.h"
#include "driverlib/sysctl.h"
#include "drivers/rtos_hw_drivers.h"
#include "utils/uartstdio.h"
/*-----------------------------------------------------------*/

/*
 * Declare a variable that is used to hold the handle of the SSI2
 * interrupt task.
 */
TaskHandle_t xSSI2IntTask = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvSSITransmitTask( void *pvParameters );
static void prvSSIReceiveTask( void *pvParameters );

/*
 * Called by main() to create the SSI tasks.
 */
void vSSITaskCreate( void );

/* 
 * Configure the SSI peripheral for both Master and Slave operation.
 */
static void prvConfigureSSIMaster( void );
static void prvConfigureSSISlave( void );
/*-----------------------------------------------------------*/

void vSSITaskCreate( void )
{
    /* Configure the SSI0 peripheral for Master Mode at 2MHz. */
    prvConfigureSSIMaster();

    /* Configure the SSI2 peripheral for Slave Mode at 2MHz. */
    prvConfigureSSISlave();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name SSITX task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - No parameter passed to the task
     *  - The priority assigned to the task.
     *  - The task handle is NULL */
    xTaskCreate( prvSSITransmitTask,
                 "SSITX",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 2,
                 NULL );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name SSIRX task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - No parameter passed to the task
     *  - The priority assigned to the task.
     *  - The task handle is NULL */
    xTaskCreate( prvSSIReceiveTask,
                 "SSIRX",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 &xSSI2IntTask );
}
/*-----------------------------------------------------------*/

static void prvSSITransmitTask( void *pvParameters )
{
uint32_t pui32DataTx0[4] = {'F', 'r', 'e', 'e'};
uint32_t pui32DataTx2[4] = {'R', 'T', 'O', 'S'};
uint32_t ui32Index;

    ui32Index = 0;

    UARTprintf("SSI0 Sending:\n  ");
    UARTprintf("'F' 'r' 'e' 'e' \n");
    UARTprintf("SSI2 Sending:\n  ");
    UARTprintf("'R' 'T' 'O' 'S' \n");

    for( ;; )
    {
        /* Prepare data to send from SSI1. */
        SSIDataPut(SSI2_BASE, pui32DataTx2[ui32Index]);

        /* Send the data using the "blocking" put function.  This function
         * will wait until there is room in the send FIFO before returning.
         * This allows you to assure that all the data you send makes it into
         * the send FIFO. */
        SSIDataPut(SSI0_BASE, pui32DataTx0[ui32Index]);

        /* Increment the index. */
        ui32Index++;

        if (ui32Index == 4)
        {
            /* Delete the task now that data has been sent. */
            vTaskDelete(NULL);
        }
    }
}
/*-----------------------------------------------------------*/

static void prvSSIReceiveTask( void *pvParameters )
{
unsigned long ulEventsToProcess;
uint32_t pui32DataRx0[4];
uint32_t pui32DataRx2[4];
uint32_t ui32Index;

    for( ;; )
    {
        /* Wait to receive a notification sent directly to this task from the
         * interrupt service routine. */
        ulEventsToProcess = ulTaskNotifyTake( pdTRUE, portMAX_DELAY );

        if (ulEventsToProcess != 0)
        {
            /* There should be four bytes received. */
            for(ui32Index = 0; ui32Index < 4; ui32Index++)
            {
                /* Receive the data using the "blocking" Get function.  This function
                 * will wait until there is data in the receive FIFO before returning. */
                SSIDataGet(SSI0_BASE, &pui32DataRx0[ui32Index]);
                SSIDataGet(SSI2_BASE, &pui32DataRx2[ui32Index]);

                /* Since we are using 8-bit data, mask off the MSB. */
                pui32DataRx0[ui32Index] &= 0x00FF;
                pui32DataRx2[ui32Index] &= 0x00FF;
            }

            /* Display the data that SSI0 received. */
            UARTprintf("SSI0 Received:\n  ");
            UARTprintf("'%c' '%c' '%c' '%c'\n", pui32DataRx0[0],
                       pui32DataRx0[1], pui32DataRx0[2], pui32DataRx0[3]);

            /* Display the data that SSI2 received. */
            UARTprintf("SSI2 Received:\n  ");
            UARTprintf("'%c' '%c' '%c' '%c'\n\n", pui32DataRx2[0],
                       pui32DataRx2[1], pui32DataRx2[2], pui32DataRx2[3]);

            /* Delete the task now that data has been printed. */
            vTaskDelete(NULL);
        }
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureSSIMaster( void )
{
uint32_t ui32DummyRead;

    /* The SSI0 peripheral must be enabled for use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI0);

    /* This example uses Port A[5:2] for the SSI0 pins.  The GPIO port needs to
     * be enabled so those pins can be used. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    /* Configure the GPIO settings for the SSI pins.  This function also gives
     * control of these pins to the SSI hardware.  Consult the data sheet to
     * see which functions are allocated per pin. */

    /* Configure the pin muxing for SSI0 functions on PA[5:2].
     * The GPIO pins are assigned as follows:
     *   PA5 - SSI0TX
     *   PA4 - SSI0RX
     *   PA3 - SSI0Fss
     *   PA2 - SSI0CLK */
    GPIOPinConfigure(GPIO_PA2_SSI0CLK);
    GPIOPinConfigure(GPIO_PA3_SSI0FSS);
    GPIOPinConfigure(GPIO_PA4_SSI0RX);
    GPIOPinConfigure(GPIO_PA5_SSI0TX);

    GPIOPinTypeSSI(GPIO_PORTA_BASE, GPIO_PIN_5 | GPIO_PIN_4 | GPIO_PIN_3 |
                   GPIO_PIN_2);

    /* Configure and enable the SSI0 port for SPI master mode.  Use SSI0,
     * system clock supply, idle clock level low and active low clock in legacy
     * Freescale SPI mode, master mode, 2MHz SSI frequency, and 8-bit data. For
     * SPI mode, you can set the polarity of the SSI clock when the SSI unit is
     * idle.  You can also configure what clock edge you want to capture data
     * on.  Please reference the device datasheet for more information on the
     * different SPI modes. */
    SSIConfigSetExpClk(SSI0_BASE, configCPU_CLOCK_HZ, SSI_FRF_MOTO_MODE_0,
                       SSI_MODE_MASTER, 2000000, 8);

    /* Enable the SSI0 module. */
    SSIEnable(SSI0_BASE);

    /* Read any residual data from the SSI port.  This makes sure the receive
     * FIFOs are empty, so we don't read any unwanted junk.  This is done here
     * because the SPI SSI mode is full-duplex, which allows you to send and
     * receive at the same time.  The SSIDataGetNonBlocking function returns
     * "true" when data was returned, and "false" when no data was returned.
     * The "non-blocking" function checks if there is any data in the receive
     * FIFO and does not "hang" if there isn't. */
    while(SSIDataGetNonBlocking(SSI0_BASE, &ui32DummyRead))
    {
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureSSISlave( void )
{
uint32_t ui32DummyRead;

    /* The SSI2 peripheral must be enabled for use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI2);

    /* Configure the GPIO settings for the SSI pins.  This function also gives
     * control of these pins to the SSI hardware.  Consult the data sheet to
     * see which functions are allocated per pin. */

    /* Configure the pin muxing for SSI2 functions on PB[7:4].
     * The GPIO pins are assigned as follows:
     *   PB7 - SSI2TX
     *   PB6 - SSI2RX
     *   PB5 - SSI2Fss
     *   PB4 - SSI2CLK */
    GPIOPinConfigure(GPIO_PB4_SSI2CLK);
    GPIOPinConfigure(GPIO_PB5_SSI2FSS);
    GPIOPinConfigure(GPIO_PB6_SSI2RX);
    GPIOPinConfigure(GPIO_PB7_SSI2TX);

    GPIOPinTypeSSI(GPIO_PORTB_BASE, GPIO_PIN_7 | GPIO_PIN_6 | GPIO_PIN_5 |
                   GPIO_PIN_4);

    /* Configure and enable the SSI2 port for SPI slave mode with matching
     * SPI mode, clock speed, and data size parameters as the master. */
    SSIConfigSetExpClk(SSI2_BASE, configCPU_CLOCK_HZ, SSI_FRF_MOTO_MODE_0,
                       SSI_MODE_SLAVE, 2000000, 8);

    /* Enable SSI2 interrupt on RX FIFO half full or more. */
    SSIIntEnable(SSI2_BASE, SSI_RXFF);

    /* Enable the SSI2 interrupt in the NVIC. */
    IntEnable(INT_SSI2);

    /* Enable the SSI2 module. */
    SSIEnable(SSI2_BASE);

    /* Read any residual data from the SSI port.  This makes sure the receive
     * FIFOs are empty, so we don't read any unwanted junk.  This is done here
     * because the SPI SSI mode is full-duplex, which allows you to send and
     * receive at the same time.  The SSIDataGetNonBlocking function returns
     * "true" when data was returned, and "false" when no data was returned.
     * The "non-blocking" function checks if there is any data in the receive
     * FIFO and does not "hang" if there isn't. */
    while(SSIDataGetNonBlocking(SSI2_BASE, &ui32DummyRead))
    {
    }
}
/*-----------------------------------------------------------*/

void xSSI2Handler( void )
{
BaseType_t xHigherPriorityTaskWoken;
uint32_t ui32Status;

    /* Read SSIMIS (SSI Masked Interrupt Status). */
    ui32Status = SSIIntStatus(SSI2_BASE, true);

    /* Clear the SSI interrupt. */
    SSIIntClear(SSI2_BASE, ui32Status);

    /* Turn off the RX FIFO interrupt. */
    SSIIntDisable(SSI2_BASE, SSI_RXFF);

    /* The xHigherPriorityTaskWoken parameter must be initialized to pdFALSE as
     * it will get set to pdTRUE inside the interrupt safe API function if a
     * context switch is required. */
    xHigherPriorityTaskWoken = pdFALSE;

    /* Send a notification directly to the task to which interrupt processing
     * is being deferred. */
    vTaskNotifyGiveFromISR( xSSI2IntTask, &xHigherPriorityTaskWoken );

    /* This FreeRTOS API call will handle the context switch if it is required
     * or have no effect if that is not needed. */
    portYIELD_FROM_ISR( xHigherPriorityTaskWoken );
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
        configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
        added here, but the tick hook is called from an interrupt context, so
        code must not attempt to block, and only the interrupt safe FreeRTOS API
        functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
        executed here. */
}


